///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include <Windows.H>
#include <Stdio.H>
#include <Stdlib.H>
#include <CommCtrl.H>

#include "Graph.H"

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void DrawGraph (HWND hWnd, DWORD BGColor)
{
    RECT rc;
    HPEN hPen1;
    HPEN hPen2;
    HBITMAP MemBM;
    HBITMAP OldBM;
    HBRUSH black_brush;
    HDC MemDC;
    HDC ScreenDC;
    HGDIOBJ GdiObj;
    GraphProps *PropSet;
    long WindowSizeX;
    long WindowSizeY;
    int x;
    long y;
    long SpacingX;
    long SpacingY;

    GetClientRect(hWnd, &rc);
    PropSet = (GraphProps *) GetProp(hWnd, "GraphProps");

    if(PropSet == NULL)
        return;

    WindowSizeX = rc.right-rc.left;
    WindowSizeY = rc.bottom-rc.top;

    black_brush = CreateSolidBrush(BGColor);

    hPen1 = CreatePen(PS_SOLID, 1, PropSet->GridColor);
    hPen2 = CreatePen(PS_SOLID, 1, PropSet->GraphColor);

    ScreenDC = GetDC(hWnd);
    MemDC = CreateCompatibleDC(ScreenDC);

    //Create a bitmap big enough for the client rectangle
    MemBM = CreateCompatibleBitmap(ScreenDC, rc.right-rc.left, rc.bottom-rc.top);
    OldBM = (HBITMAP) SelectObject(MemDC, MemBM);

    // Draw the Grid
    FillRect(MemDC, &rc, black_brush);
    SpacingX = 10;
    SpacingY = 10;
    SelectObject(MemDC, hPen1);

    for( x = 0; x <= WindowSizeX; x = x + SpacingX)
    {
        MoveToEx(MemDC, x, 0, NULL);
        LineTo(MemDC, x, WindowSizeY);
    }

    for( y = 0; y <= WindowSizeY; y = y + SpacingY)
    {
        MoveToEx(MemDC, 0, y, NULL);
        LineTo(MemDC, WindowSizeX, y);
    }

    // Now to draw the graph in one pass
    SelectObject(MemDC, hPen2);
    Polyline(MemDC,PropSet->GraphPtr, (WindowSizeX/PropSet->StepSize) + 1);

    BitBlt(ScreenDC, rc.left, rc.top, rc.right-rc.left, rc.bottom-rc.top, MemDC, 0, 0, SRCCOPY);

    GdiObj = SelectObject(MemDC,OldBM);

    // Clean up after ourselves to prevent resource leaks.
    ReleaseDC(hWnd,ScreenDC);
    DeleteDC(MemDC);
    DeleteObject(black_brush);
    DeleteObject(hPen1);
    DeleteObject(hPen2);
    DeleteObject(OldBM);
    DeleteObject(MemBM);
    DeleteObject(GdiObj);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void InitGraph(HWND hWnd, COLORREF GRColor, COLORREF GridColor, UCHAR StepSize, BOOL Direction)
{
    GraphProps *GraphPropSet;
    POINT *GPtr;
    RECT rc;
    POINT * LGPtr;
    long lx;

    GraphPropSet = (GraphProps*) calloc(sizeof(GraphProps), 1);
    GetClientRect(hWnd, &rc);
    GPtr = (POINT*) calloc( sizeof(POINT) * (rc.right-rc.left / StepSize) + 1, 1);

    if(Direction == GRAPH_LEFT)
    {
        for( lx = 0; lx <= ((rc.right-rc.left) / StepSize); lx++)
        {
            GPtr[lx].x = lx * StepSize;
            GPtr[lx].y = rc.bottom - rc.top;
        }
    }

    if(Direction == GRAPH_RIGHT)
    {
        for( lx = ((rc.right-rc.left) / StepSize); lx >= 0; lx = lx + -1)
        {
            GPtr[lx].x =lx * StepSize;
            GPtr[lx].y =rc.bottom - rc.top;
        }
    }

    // This is to remove the Slow calloc call from UpdateGraph()
    LGPtr = (POINT *) calloc( sizeof(POINT) * (rc.right-rc.left / StepSize) + 1, 1);
    memset(LGPtr, 0, sizeof(POINT)*(rc.right-rc.left / StepSize) + 1);

    // Add a few more bells and whistles to the graph
    GraphPropSet->GraphPtr   = GPtr;
    GraphPropSet->LGraphPtr  = LGPtr;
    GraphPropSet->GraphColor = GRColor;
    GraphPropSet->GridColor  = GridColor;
    GraphPropSet->Frozen     = FALSE;
    GraphPropSet->StepSize   = StepSize;
    GraphPropSet->Direction  = Direction;
    SetProp(hWnd, "GraphProps", GraphPropSet);

    ShowWindow(hWnd, SW_SHOW);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void UpdateGraph(HWND hWnd, int UpdtVal, int MaxVal)
{
    RECT rc;
    long WindowSizeX;
    long WindowSizeY;

    GetClientRect(hWnd,&rc);
    WindowSizeX=rc.right-rc.left;
    WindowSizeY=rc.bottom-rc.top;

    POINT*  pntarray;
    POINT*  lpointarray;
    GraphProps*  PropSet;

    int lx;
    int locSteps;
    if(UpdtVal>MaxVal)
    {
      UpdtVal=MaxVal;
    }

    PropSet = (GraphProps*)GetProp(hWnd,"GraphProps");

    if(PropSet == NULL)
        return;

    if(PropSet->Frozen==TRUE)
    {
      return;
    }

    pntarray=PropSet->GraphPtr;
    lpointarray=PropSet->LGraphPtr;
    locSteps=PropSet->StepSize;
    if(PropSet->Direction==GRAPH_LEFT)
    {
        lpointarray->y=WindowSizeY-((WindowSizeY*UpdtVal)/MaxVal);
        for( lx=1;lx<=(WindowSizeX/PropSet->StepSize);lx++)
        {
            (lpointarray+lx)->y=(pntarray+(lx-1))->y;
        }

        for( lx=0;lx<=(WindowSizeX/PropSet->StepSize);lx++)
        {
            (pntarray+lx)->y=(lpointarray+lx)->y;
        }
    }
    else
    {
        (lpointarray+(WindowSizeX/PropSet->StepSize))->y=WindowSizeY-((WindowSizeY*UpdtVal)/MaxVal);

        for( lx=(WindowSizeX/PropSet->StepSize)-1;lx>=0;lx=lx+-1)
        {
            (lpointarray+lx)->y=(pntarray+(lx+1))->y;
        }

        for( lx=(WindowSizeX/PropSet->StepSize);lx>=0;lx=lx+-1)
        {
            (pntarray+lx)->y=(lpointarray+lx)->y;
        }
    }
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void CloseGraph(HWND hWnd)
{
    GraphProps *tmp;
    VOID *DestPtr;

    tmp = (GraphProps*)GetProp(hWnd,"GraphProps");
    DestPtr = tmp->GraphPtr;
    free(DestPtr);
    DestPtr = tmp->LGraphPtr;
    free(DestPtr);
    free(tmp);
    RemoveProp(hWnd,"GraphProps");
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void FreezeGraph(HWND hWnd, BOOL FreezeStatus)
{
    GraphProps *Gprops;
    Gprops = (GraphProps*) GetProp(hWnd, "GraphProps");

    if(FreezeStatus) Gprops->Frozen=TRUE;
    else Gprops->Frozen=FALSE;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL IsGraphFrozen(HWND hWnd)
{
    GraphProps *Gprops;
    Gprops = (GraphProps*) GetProp(hWnd, "GraphProps");
    return Gprops->Frozen;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

